using System;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class defines response status codes used in VistALink protocol.
	/// Works as a "string enum".
	/// </summary>
	/// <remarks>
	/// Be careful changing underlying string values for this "enum": 
	/// they are mapped directly to VistALink messages XML representation 
	/// (used in XML serialization/deserialization). 
	/// </remarks>
	public sealed class ResponseStatus
	{
		/// <summary>
		/// Indicates success of the previous request.
		/// </summary>
		public static readonly ResponseStatus Success = new ResponseStatus( "success" ); 
		
		/// <summary>
		/// Indicates that previous request has failed.
		/// </summary>
		public static readonly ResponseStatus Failure = new ResponseStatus( "failure" );

		/// <summary>
		/// Indicates that previous request was partially successful.
		/// </summary>
		public static readonly ResponseStatus PartialSuccess = new ResponseStatus( "partialSuccess" );

		private readonly string _statusCode;

		/// <summary>
		/// The only private constructor. It's called upon 
		/// static initialization of "enum" members.
		/// </summary>
		/// <param name="statusCode">Status code to assign</param>
		private ResponseStatus( string statusCode )
		{
			_statusCode = statusCode;
		}

		/// <summary>
		/// Override for standard ToString() method.
		/// </summary>
		/// <returns>
		///		String representation of response status as 
		///		defined in VistALink protocol specification
		/// </returns>
		public override string ToString()
		{
			return _statusCode;
		}

		/// <summary>
		/// Parses a given string and returns appropriate response status object.
		/// Will throw ArgumentOutOfRange exception if input string cannot be parsed.
		/// </summary>
		/// <param name="statusCodeString">
		///		String representation for response status code 
		///		as defined in VistALink protocol.
		///	</param>
		/// <returns>
		///		Appropriate response status object.
		///	</returns>
		public static ResponseStatus Parse( string statusCodeString ) 
		{
			if( statusCodeString == null )
				throw( new ArgumentNullException( "statusCodeString" ) );

			switch( statusCodeString )
			{
				case "success":			return Success; 
				case "failure":			return Failure; 
				case "partialSuccess" :	return PartialSuccess;
				default:
					throw( new StringParseException( SR.Exceptions.StringEnumValueNotFound( "Response status", statusCodeString ) ) );
			}
		}
	}
}
